<?php
if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

function portainer_MetaData()
{
    return [
        'DisplayName' => 'Portainer EE SaaS Module (Nestict)',
        'APIVersion' => '1.1',
        'RequiresServer' => false,
    ];
}

function portainer_ConfigOptions()
{
    return [
        'API URL' => [
            'Type' => 'text',
            'Size' => '60',
            'Default' => 'https://www.portainer.nestict.africa/api',
            'Description' => 'Portainer API endpoint (include /api).',
        ],
        'API Token' => [
            'Type' => 'password',
            'Size' => '80',
            'Description' => 'Portainer API access token (do NOT include "Bearer ").',
        ],
        'Default Image' => [
            'Type' => 'dropdown',
            'Options' => 'wordpress:latest,joomla:latest,nextcloud:latest,nginx:latest,php:8.0-apache,php:8.1-apache,php:8.2-apache,php:8.3-apache,php:8.4-apache',
            'Default' => 'wordpress:latest',
            'Description' => 'Default container image for this product.',
        ],
        'Endpoint ID' => [
            'Type' => 'text',
            'Size' => '6',
            'Default' => '1',
            'Description' => 'Portainer Endpoint ID (check Portainer → Environments).',
        ],
        'Network Mode' => [
            'Type' => 'dropdown',
            'Options' => 'bridge,host,none',
            'Default' => 'bridge',
            'Description' => 'Docker network mode for created containers.',
        ],
    ];
}

/**
 * Create account - pull image if needed, then create and start a container via Portainer EE API
 */
function portainer_CreateAccount($params)
{
    $apiUrl = rtrim($params['configoption1'], '/');
    $token = trim($params['configoption2']);
    $image = $params['configoption3'];
    $endpoint = trim($params['configoption4']) ?: '1';
    $network = $params['configoption5'] ?: 'bridge';
    $serviceId = $params['serviceid'];

    $clientName = isset($params['clientsdetails']['firstname']) ? preg_replace('/[^a-z0-9\-]/i', '', strtolower($params['clientsdetails']['firstname'])) : 'client';
    $containerName = "nestict-{$serviceId}-{$clientName}";

    // Step 1: Pull the image if not local (handles "No such image" errors)
    $imageParts = explode(':', $image);
    $imageName = $imageParts[0];
    $imageTag = $imageParts[1] ?? 'latest';
    $pullUrl = $apiUrl . "/endpoints/{$endpoint}/docker/images/create?fromImage={$imageName}&tag={$imageTag}";
    logModuleCall('Portainer', 'PullImageRequest', $pullUrl, ['image' => $image], null);
    $pullResp = portainer_api_request($pullUrl, $token, [], 'POST');
    if (isset($pullResp['error']) && !strpos($pullResp['error'], 'already exists') && (isset($pullResp['http_code']) && $pullResp['http_code'] >= 400)) {
        logModuleCall('Portainer', 'PullImageError', $pullUrl, ['image' => $image], $pullResp);
        return "Failed to pull image '{$image}': " . json_encode($pullResp) . ". Try pulling manually in Portainer UI.";
    }
    // Brief pause to ensure pull completes (API responds after pull)
    sleep(5);

    // Step 2: Create container
    $createBody = [
        'Image' => $image,
        'name' => $containerName,
        'HostConfig' => [
            'NetworkMode' => $network,
            'RestartPolicy' => ['Name' => 'unless-stopped'],
        ],
        'Env' => [
            "WHMCS_CLIENT_ID={$params['clientsdetails']['userid']}",
            "WHMCS_CONTACT_EMAIL={$params['clientsdetails']['email']}",
            "WHMCS_SERVICE_ID={$serviceId}"
        ]
    ];

    $createUrl = $apiUrl . "/endpoints/{$endpoint}/docker/containers/create";
    logModuleCall('Portainer', 'CreateContainerRequest', $createUrl, $createBody, null);

    $response = portainer_api_request($createUrl, $token, $createBody, 'POST');

    if (isset($response['error'])) {
        logModuleCall('Portainer', 'CreateContainerError', $createUrl, $createBody, $response);
        return "Failed to create container: " . $response['error'];
    }

    if (isset($response['Id']) && $response['Id']) {
        $containerId = $response['Id'];
        $startUrl = $apiUrl . "/endpoints/{$endpoint}/docker/containers/{$containerId}/start";
        $startResp = portainer_api_request($startUrl, $token, [], 'POST');
        if (isset($startResp['error']) || (isset($startResp['http_code']) && $startResp['http_code'] >= 400)) {
            // Rollback: delete the container on start failure
            $deleteUrl = $apiUrl . "/endpoints/{$endpoint}/docker/containers/{$containerId}?force=1";
            portainer_api_request($deleteUrl, $token, [], 'DELETE');
            logModuleCall('Portainer', 'StartContainerErrorWithRollback', $startUrl, [], $startResp);
            return "Failed to start container: " . json_encode($startResp);
        }
        logModuleCall('Portainer', 'StartContainer', $startUrl, [], $startResp);
        return 'success';
    }

    if (isset($response['message'])) {
        logModuleCall('Portainer', 'CreateContainerResponse', $createUrl, $createBody, $response);
        return "Failed to create container: " . json_encode($response);
    }

    logModuleCall('Portainer', 'CreateContainerUnknown', $createUrl, $createBody, $response);
    return 'Failed to create container: Unknown error';
}

function portainer_SuspendAccount($params)
{
    return portainer_container_action($params, 'stop');
}

function portainer_UnsuspendAccount($params)
{
    return portainer_container_action($params, 'start');
}

function portainer_TerminateAccount($params)
{
    return portainer_container_action($params, 'delete');
}

function portainer_ClientArea($params)
{
    $apiUrl = rtrim($params['configoption1'], '/');
    $token = trim($params['configoption2']);
    $endpoint = trim($params['configoption4']) ?: '1';
    $serviceId = $params['serviceid'];
    $clientName = isset($params['clientsdetails']['firstname']) ? preg_replace('/[^a-z0-9\-]/i', '', strtolower($params['clientsdetails']['firstname'])) : 'client';
    $containerName = "nestict-{$serviceId}-{$clientName}";

    $containers = portainer_api_request($apiUrl . "/endpoints/{$endpoint}/docker/containers/json?all=1", $token, [], 'GET');

    $info = "Container not found or not yet provisioned.";
    if (is_array($containers)) {
        foreach ($containers as $c) {
            if (in_array("/{$containerName}", $c['Names'])) {
                $state = $c['State'] ?? 'unknown';
                $image = $c['Image'] ?? '';
                $id = $c['Id'] ?? '';
                $ports = json_encode($c['Ports'] ?? '');
                $info = "<strong>Status:</strong> {$state}<br><strong>Image:</strong> {$image}<br><strong>ID:</strong> {$id}<br><strong>Ports:</strong> {$ports}";
                break;
            }
        }
    }

    return [
        'tabOverviewReplacementTemplate' => 'portainer_clientarea.tpl',
        'templateVariables' => ['info' => $info],
    ];
}

/* ---------- Helper functions ---------- */

function portainer_api_request($url, $token, $payload = [], $method = 'GET')
{
    $ch = curl_init();

    $headers = [
        "X-API-Key: {$token}",  // Fixed: Use X-API-Key for Portainer Access Tokens
        "Content-Type: application/json",
    ];

    $opts = [
        CURLOPT_URL => $url,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 300,  // Increased for image pulls (5 min)
        CURLOPT_SSL_VERIFYPEER => false,
    ];

    if (strtoupper($method) === 'POST') {
        $opts[CURLOPT_POST] = true;
        if (!empty($payload)) {
            $opts[CURLOPT_POSTFIELDS] = json_encode($payload);
        }
    } elseif (strtoupper($method) === 'DELETE') {
        $opts[CURLOPT_CUSTOMREQUEST] = 'DELETE';
        if (!empty($payload)) {
            $opts[CURLOPT_POSTFIELDS] = json_encode($payload);
        }
    } else {
        if (!empty($payload) && is_array($payload)) {
            $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($payload);
            $opts[CURLOPT_URL] = $url;
        }
    }

    curl_setopt_array($ch, $opts);
    $resp = curl_exec($ch);
    $err = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err) {
        return ['error' => $err];
    }

    $decoded = json_decode($resp, true);
    if ($decoded === null) {
        return ['raw' => $resp, 'http_code' => $http_code];
    }
    $decoded['http_code'] = $http_code;  // Always include HTTP code for error checking
    return $decoded;
}

function portainer_container_action($params, $action)
{
    $apiUrl = rtrim($params['configoption1'], '/');
    $token = trim($params['configoption2']);
    $endpoint = trim($params['configoption4']) ?: '1';
    $serviceId = $params['serviceid'];
    $clientName = isset($params['clientsdetails']['firstname']) ? preg_replace('/[^a-z0-9\-]/i', '', strtolower($params['clientsdetails']['firstname'])) : 'client';
    $containerName = "nestict-{$serviceId}-{$clientName}";

    $listUrl = $apiUrl . "/endpoints/{$endpoint}/docker/containers/json?all=1";
    $containers = portainer_api_request($listUrl, $token, [], 'GET');

    if (isset($containers['error'])) {
        logModuleCall('Portainer', 'ListContainersError', $listUrl, [], $containers);
        return "Error listing containers: " . $containers['error'];
    }

    if (is_array($containers)) {
        foreach ($containers as $c) {
            if (in_array("/{$containerName}", $c['Names'])) {
                $id = $c['Id'];
                if ($action === 'delete') {
                    $url = $apiUrl . "/endpoints/{$endpoint}/docker/containers/{$id}?force=1";  // Fixed: Use ?force=1 for consistency
                    $result = portainer_api_request($url, $token, [], 'DELETE');
                    logModuleCall('Portainer', 'DeleteContainer', $url, [], $result);
                    return 'success';
                } else {
                    $url = $apiUrl . "/endpoints/{$endpoint}/docker/containers/{$id}/{$action}";
                    $result = portainer_api_request($url, $token, [], 'POST');
                    logModuleCall('Portainer', ucfirst($action).'Container', $url, [], $result);
                    return 'success';
                }
            }
        }
    }

    return 'Container not found';
}
?>