<?php
/**
 * Nestict M-Pesa WHMCS Gateway (STK Push) - main module file
 * Place at: /modules/gateways/nestictmpesa.php
 *
 * Requires: WHMCS 7.x+ (uses Capsule if available)
 *
 * NOTE: This file expects the callback handler to be placed at:
 * /modules/gateways/callback/nestictmpesa.php
 */

if (!defined('WHMCS')) {
    die('This file cannot be accessed directly');
}

use WHMCS\Database\Capsule;

function nestictmpesa_MetaData() {
    return [
        'DisplayName' => 'Nestict M-Pesa (STK Push) - Nestict',
        'APIVersion'  => '1.1',
        'DisableLocalCreditCardInput' => true,
        'TokenisedStorage' => false,
    ];
}

function nestictmpesa_config() {
    return [
        'FriendlyName' => [
            'Type' => 'System',
            'Value' => 'Nestict M-Pesa (STK Push)'
        ],
        'shortcode' => [
            'FriendlyName' => 'Paybill Number',
            'Type' => 'text',
            'Size' => '20',
            'Description' => 'Your Paybill (Business Short Code) e.g. 123456'
        ],
        'consumerkey' => [
            'FriendlyName' => 'Consumer Key',
            'Type' => 'text',
            'Size' => '50',
        ],
        'consumersecret' => [
            'FriendlyName' => 'Consumer Secret',
            'Type' => 'password',
            'Size' => '50',
        ],
        'passkey' => [
            'FriendlyName' => 'Passkey',
            'Type' => 'password',
            'Size' => '80',
            'Description' => 'Lipa Na M-Pesa Online (STK) Passkey for your Paybill'
        ],
        'environment' => [
            'FriendlyName' => 'Environment',
            'Type' => 'dropdown',
            'Options' => ['production' => 'Production', 'sandbox' => 'Sandbox'],
            'Default' => 'production',
        ],
        'callbackurl' => [
            'FriendlyName' => 'Callback URL',
            'Type' => 'text',
            'Size' => '80',
            'Description' => 'The callback URL to register with Daraja. Example: https://billing.nestict.net/modules/gateways/callback/nestictmpesa.php'
        ],
        'notes' => [
            'FriendlyName' => 'Notes',
            'Type' => 'textarea',
            'Description' => 'Optional notes for admin'
        ],
    ];
}

function nestictmpesa_link($params) {
    // params passed from WHMCS
    $invoiceId = (int) $params['invoiceid'];
    $invoiceNum = $params['invoicenum'] ?: (isset($params['invoice_num']) ? $params['invoice_num'] : $invoiceId);
    $amount = number_format((float)$params['amount'], 2, '.', '');
    $currency = $params['currency'];
    $shortcode = $params['shortcode'];
    $env = isset($params['environment']) ? $params['environment'] : 'production';
    $callbackUrl = !empty($params['callbackurl']) ? $params['callbackurl'] : (isset($params['systemurl']) ? rtrim($params['systemurl'], '/') . '/modules/gateways/callback/nestictmpesa.php' : '');

    $langPayNow = $params['langpaynow'];

    $html = '';
    $html .= '<div class="gateway-instructions">';
    $html .= '<p><strong>Paybill: ' . htmlspecialchars($shortcode) . '</strong><br>Amount: <strong>' . htmlspecialchars($amount) . ' ' . htmlspecialchars($currency) . '</strong></p>';
    $html .= '</div>';

    // STK Push form (phone input)
    $html .= '<form method="post" action="' . htmlspecialchars($_SERVER['PHP_SELF']) . '?id=' . $invoiceId . '">';
    $html .= '<input type="hidden" name="invoice_id" value="' . htmlspecialchars($invoiceId) . '">';
    $html .= '<input type="tel" name="phone" placeholder="07XXXXXXXX" required style="border-radius:5px;padding:8px;width:200px;" />';
    $html .= '<input type="submit" name="do_stk" value="' . htmlspecialchars($langPayNow) . '" style="margin-left:10px;padding:8px 14px;" />';
    $html .= '</form>';

    // manual transaction entry (fallback)
    $html .= '<hr /><h4>And  enter M-Pesa Transaction Code</h4>';
    $html .= '<form method="post" action="' . htmlspecialchars($_SERVER['PHP_SELF']) . '?id=' . $invoiceId . '">';
    $html .= '<input type="hidden" name="manual_invoice_id" value="' . htmlspecialchars($invoiceId) . '">';
    $html .= '<input type="text" name="manual_trans_id" placeholder="Enter M-Pesa Transaction ID" style="border-radius:5px;padding:8px;width:240px;" />';
    $html .= '<input type="submit" name="do_manual" value="Record Payment" style="margin-left:10px;padding:8px 14px;" />';
    $html .= '</form>';

    // Handle STK Push submission
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['do_stk']) && !empty($_POST['phone'])) {
        $phone = preg_replace('/[^0-9+]/', '', $_POST['phone']);
        // Normalize common formats: 07XXXXXXXX -> 2547XXXXXXXX
        if (preg_match('/^0[0-9]{9}$/', $phone)) {
            $phone = '254' . substr($phone, 1);
        } elseif (preg_match('/^\+254[0-9]{9}$/', $phone)) {
            $phone = substr($phone, 1);
        }
        if (!preg_match('/^254[0-9]{9}$/', $phone)) {
            $html = '<div class="alert alert-danger">Please enter a valid phone number in Kenyan format (e.g. 07XXXXXXXX).</div>' . $html;
        } else {
            // check if API keys present
            if (!empty($params['consumerkey']) && !empty($params['consumersecret']) && !empty($params['passkey'])) {
                try {
                    $res = nestictmpesa_send_stk_push($invoiceId, $invoiceNum, $amount, $phone, $params);
                    if ($res['success']) {
                        $html = '<div class="alert alert-success">STK Push initiated to ' . htmlspecialchars($phone) . '. Follow the prompt on your phone to complete payment.</div>' . $html;
                    } else {
                        $html = '<div class="alert alert-danger">STK Push failed: ' . htmlspecialchars($res['message']) . '</div>' . $html;
                    }
                } catch (Exception $e) {
                    logModuleCall('nestictmpesa', 'stkpush_exception', ['post' => $_POST, 'params' => $params], $e->getMessage());
                    $html = '<div class="alert alert-danger">An error occurred initiating STK Push. Check module logs.</div>' . $html;
                }
            } else {
                $html = '<div class="alert alert-info">API credentials not configured. Please use the manual transaction code form below to record payment.</div>' . $html;
            }
        }
    }

    // Handle manual transaction record
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['do_manual']) && !empty($_POST['manual_trans_id'])) {
        $mtrans = strtoupper(trim($_POST['manual_trans_id']));
        try {
            $dup = 0;
            if (class_exists('WHMCS\\Database\\Capsule')) {
                $dup = Capsule::table('tblaccounts')->where('transid', $mtrans)->count();
            }
            if ($dup > 0) {
                $html = '<div class="alert alert-warning">That transaction code has already been used.</div>' . $html;
            } else {
                addInvoicePayment($invoiceId, $mtrans, $amount, 0, 'nestictmpesa');
                logModuleCall('nestictmpesa', 'manual_record', ['invoice' => $invoiceId, 'trans' => $mtrans], 'success');
                header('Location: viewinvoice.php?id=' . $invoiceId . '&mpesa_success=1');
                exit;
            }
        } catch (Exception $e) {
            logModuleCall('nestictmpesa', 'manual_record_error', ['exception' => $e->getMessage()], 'failure');
            $html = '<div class="alert alert-danger">Failed to record manual payment. Check logs.</div>' . $html;
        }
    }

    return $html;
}

/**
 * Request OAuth token (Daraja)
 */
function nestictmpesa_get_oauth_token($consumerKey, $consumerSecret, $env = 'production') {
    $url = $env === 'sandbox'
        ? 'https://sandbox.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials'
        : 'https://api.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials';

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_USERPWD, $consumerKey . ':' . $consumerSecret);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    $resp = curl_exec($ch);
    $errno = curl_errno($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($errno) {
        throw new Exception('OAuth request error: ' . $err);
    }
    $data = json_decode($resp, true);
    if (!isset($data['access_token'])) {
        throw new Exception('Invalid OAuth response: ' . $resp);
    }
    return $data['access_token'];
}

/**
 * Send STK Push to Daraja
 */
function nestictmpesa_send_stk_push($invoiceId, $invoiceNum, $amount, $phone, $params) {
    // create mapping table if not exists
    try {
        if (class_exists('WHMCS\\Database\\Capsule')) {
            if (!Capsule::schema()->hasTable('tblmpesarequests')) {
                Capsule::schema()->create('tblmpesarequests', function ($table) {
                    $table->increments('id');
                    $table->string('merchant_request_id')->nullable();
                    $table->string('checkout_request_id')->nullable();
                    $table->integer('invoice_id');
                    $table->string('phone');
                    $table->decimal('amount', 10, 2);
                    $table->string('status')->default('Pending');
                    $table->text('response')->nullable();
                    $table->timestamp('created_at')->default(Capsule::raw('CURRENT_TIMESTAMP'));
                });
            }
        }
    } catch (Exception $e) {
        // ignore - table probably exists or permission issue will show in logs
        logModuleCall('nestictmpesa', 'create_tbl_error', [], $e->getMessage());
    }

    $consumerKey = $params['consumerkey'];
    $consumerSecret = $params['consumersecret'];
    $passkey = $params['passkey'];
    $shortcode = $params['shortcode'];
    $env = isset($params['environment']) ? $params['environment'] : 'production';
    $callbackUrl = !empty($params['callbackurl']) ? $params['callbackurl'] : (isset($params['systemurl']) ? rtrim($params['systemurl'], '/') . '/modules/gateways/callback/nestictmpesa.php' : '');

    $token = nestictmpesa_get_oauth_token($consumerKey, $consumerSecret, $env);

    $timestamp = date('YmdHis');
    $password = base64_encode($shortcode . $passkey . $timestamp);

    $requestBody = [
        'BusinessShortCode' => $shortcode,
        'Password' => $password,
        'Timestamp' => $timestamp,
        'TransactionType' => 'CustomerPayBillOnline',
        'Amount' => (int) round($amount, 0),
        'PartyA' => $phone,
        'PartyB' => $shortcode,
        'PhoneNumber' => $phone,
        'CallBackURL' => $callbackUrl,
        'AccountReference' => $invoiceNum ?: $invoiceId,
        'TransactionDesc' => 'Invoice ' . ($invoiceNum ?: $invoiceId),
    ];

    $url = $env === 'sandbox'
        ? 'https://sandbox.safaricom.co.ke/mpesa/stkpush/v1/processrequest'
        : 'https://api.safaricom.co.ke/mpesa/stkpush/v1/processrequest';

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $token,
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestBody));
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

    $response = curl_exec($ch);
    $errno = curl_errno($ch);
    $err = curl_error($ch);
    curl_close($ch);

    if ($errno) {
        logModuleCall('nestictmpesa', 'stkpush_curl_error', $requestBody, $err);
        return ['success' => false, 'message' => $err];
    }

    $decoded = json_decode($response, true);
    logModuleCall('nestictmpesa', 'stkpush_request', $requestBody, $decoded);

    if (isset($decoded['ResponseCode']) && ($decoded['ResponseCode'] === '0' || $decoded['ResponseCode'] === 0)) {
        $merchantRequestID = isset($decoded['MerchantRequestID']) ? $decoded['MerchantRequestID'] : null;
        $checkoutRequestID = isset($decoded['CheckoutRequestID']) ? $decoded['CheckoutRequestID'] : null;
        try {
            if (class_exists('WHMCS\\Database\\Capsule')) {
                Capsule::table('tblmpesarequests')->insert([
                    'merchant_request_id'  => $merchantRequestID,
                    'checkout_request_id'  => $checkoutRequestID,
                    'invoice_id' => $invoiceId,
                    'phone' => $phone,
                    'amount' => $amount,
                    'status' => 'Pending',
                    'response' => json_encode($decoded),
                ]);
            }
        } catch (Exception $e) {
            logModuleCall('nestictmpesa', 'tblmpesarequests_insert_error', ['resp' => $decoded], $e->getMessage());
        }
        return ['success' => true, 'message' => 'STK Push sent', 'response' => $decoded];
    }

    $message = isset($decoded['errorMessage']) ? $decoded['errorMessage'] : (isset($decoded['ResultDesc']) ? $decoded['ResultDesc'] : json_encode($decoded));
    return ['success' => false, 'message' => $message, 'response' => $decoded];
}